/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/vmstat.h>
#include <linux/highmem.h>
#include <linux/swap.h>

#include <asm/pgtable.h>

//#include <trace/events/generic.h>

struct gup_private_data {
    int nr;
    struct page **pages;
    int write;
};

static int gup_pte_entry(pte_t *ptep, unsigned long start, unsigned long end, struct mm_walk *walk)
{
    struct gup_private_data *private_data = (struct gup_private_data *)walk->private;
    struct page * page;
    pte_t pte = *ptep;
    if (!pte_present(pte) || pte_special(pte) || (private_data->write && !pte_write(pte)))
    {
        return private_data->nr;
    }
    page = pte_page(pte);
    get_page(page);
    private_data->pages[private_data->nr++] = page;
    return 0;
}

static int gup_pte_hole_entry(unsigned long start, unsigned long end, struct mm_walk *walk)
{
    struct gup_private_data *private_data = (struct gup_private_data *)walk->private;
    return private_data->nr;
}


int get_user_pages_fast(unsigned long start, int nr_pages, int write, struct page **pages)
{
    struct mm_struct *mm = current->mm;
    int ret;
    unsigned long page_addr = (start & PAGE_MASK);
    int nr = 0;

    struct gup_private_data private_data = {
        .nr = 0,
        .pages = pages,
        .write = write
    };

    struct mm_walk gup_walk = {
        .pte_entry = gup_pte_entry,
        .pte_hole = gup_pte_hole_entry,
        .mm = mm,
        .private = (void *)&private_data
    };

//    trace_generic("get_user_pages_fast(%lu, %d, %d, %p) start", start, nr_pages, write, pages);

    ret = walk_page_range(page_addr, page_addr + nr_pages * PAGE_SIZE, &gup_walk);
    nr = ret ? ret : nr_pages;

//    trace_generic("get_user_pages_fast(%lu, %d, %d, %p) got %d pages through fast path", start, nr_pages, write, pages, nr);
    if (nr == nr_pages)
    {
        return nr;
    }
    else
    {
        page_addr += (nr << PAGE_SHIFT);
    }

    down_read(&mm->mmap_sem);
    ret = get_user_pages(current, mm, page_addr, nr_pages - nr, write, 0, pages + nr, NULL);
    up_read(&mm->mmap_sem);

//    trace_generic("get_user_pages_fast(%lu, %d, %d, %p) finished slow path for %d pages", start, nr_pages, write, pages, nr_pages - nr);

    return (ret < 0) ? nr : (ret + nr);


}