/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifndef INST_BOOT_LOG_LOCAL_H_
#define INST_BOOT_LOG_LOCAL_H_

#include <linux/types.h>
#include <asm/div64.h>
#include <linux/spinlock.h>

#include <linux/inst_boot_log.h>

#ifdef INST_BOOT_LOG_C
#define GLOBAL(x) x
#define GLOBAL_VALUE(x, val) x = val
#else
#define GLOBAL(x) extern x
#define GLOBAL_VALUE(x, val) extern x
#endif

/******************************************************************************
 Buffer layout
 +------------+----------------------------------------------------+----------+
 |            |                                                    |          |
 |   Header   |                           Log                      | Overflow |
 |            |                          Buffer                    |   msg    |
 |            |                                                    |          |
 +------------+----------------------------------------------------+----------+
 <--------------------------------- size ------------------------------------->
 ^
 |
 base
******************************************************************************/

#define INST_BOOT_MAGIC_NUMBER                   0x5B1BFA53C9605B1EULL

#define INST_BOOT_WRAP_ON_OVERFLOW               1
#define INST_BOOT_OVERFLOW_MSG                   "OVERFLOW"
#define INST_BOOT_OVERFLOW_MSG_SIZE              (sizeof INST_BOOT_OVERFLOW_MSG)

#define INST_BOOT_LOG_HEADER_ADDR(base)          (base)
#define INST_BOOT_LOG_HEADER_SIZE                128

#define INST_BOOT_LOG_LOGBUFFER_ADDR(base)       (INST_BOOT_LOG_HEADER_ADDR(base) + INST_BOOT_LOG_HEADER_SIZE)

#define INST_BOOT_LOG_MIN_LOGBUFFER_SIZE         (INST_BOOT_LOG_HEADER_SIZE + INST_BOOT_OVERFLOW_MSG_SIZE )

#define INST_BOOT_LOG_LOGBUFFER_SIZE(size)       ((size) - INST_BOOT_LOG_MIN_LOGBUFFER_SIZE)

#define INST_BOOT_LOGGING_OVERHEAD_START         uint64_t _startcount = get_cntpct_count()
#define INST_BOOT_LOGGING_OVERHEAD_END           if(0 != inst_boot_log_phdr) \
                                                     inst_boot_log_phdr->logging_overhead += (get_cntpct_count() - _startcount)

#define MEM_1       1
#define MEM_2       2

#define INST_BOOT_LOGGING_KERNEL_PRE_TAG        "LINUX"

/* For a compressed kernel there will be 4 boot stages, namely SBL, APPSBL, DECOMPRESS, and OS.
   Whereas, for an uncompressed kernel DECOMPRESS will be skipped. */
#define BOOT_STAGES 5
struct _inst_boot_log_hdr
{   /*bytes*/
    /*  8 */ uint64_t magic_number;
    /*  8 */ uint64_t logging_overhead;
    /* 40 */ uint64_t oninitcntpctcount[BOOT_STAGES];
    /* 20 */ uint32_t cntpctfreq[BOOT_STAGES];
    /*  4 */ uint8_t* pbufbase;
    /*  4 */ uint32_t bufsize;
    /*  4 */ uint8_t* plogbuf;
    /*  4 */ uint32_t logsize;
    /*  4 */ uint32_t count_res;
    /*  1 */ uint8_t  bootstage;
    /*  1 */ uint8_t  memtype;
    /*  1 */ uint8_t  islogbuffull;
    /*  1 */ uint8_t  initdone;  /* 1-DONE */
    /*100 total */
             uint8_t  reserved[INST_BOOT_LOG_HEADER_SIZE - 100];
} __attribute__((packed));
typedef struct _inst_boot_log_hdr inst_boot_log_hdr;

GLOBAL_VALUE(inst_boot_log_hdr* inst_boot_log_phdr, 0);
GLOBAL_VALUE(bool inst_boot_log_lock_initialized, 0);
GLOBAL_VALUE(spinlock_t inst_boot_log_lock, {});

/* Internal functions */
static inline uint64_t get_cntpct_count(void)
{
    uint32_t vall = 0, valh = 0;
    __asm __volatile__("mrrc p15, 0, %0, %1, c14" : "=r" (vall), "=r" (valh));
    return ((uint64_t) valh << 32) | vall;
}

static inline uint32_t get_cntpct_freq(void)
{
    uint32_t val = 0;
    __asm __volatile__("mrc p15, 0, %0, c14, c0, 0" : "=r" (val));
    return val;
}

static inline void ibmemset(uint8_t* addr, uint32_t size, uint8_t val)
{
    uint32_t i = 0;
    for(i = 0; i < size; i++, addr++)
    {
        *addr = val;
    }

    return;
}

static inline void ibmemcpy(uint8_t* dstaddr, uint8_t* srcaddr, uint32_t size)
{
    uint32_t i = 0;
    for(i = 0; i < size; i++, dstaddr++, srcaddr++)
    {
        *dstaddr = *srcaddr;
    }

    return;
}

static inline char* get_count_res_str(void)
{
    if(INST_BOOT_RESOLUTION_ms == inst_boot_log_phdr->count_res)
    {
        return " ms";
    }
    else if(INST_BOOT_RESOLUTION_100us == inst_boot_log_phdr->count_res)
    {
        return " x100 us";
    }
    else if(INST_BOOT_RESOLUTION_us == inst_boot_log_phdr->count_res)
    {
        return " us";
    }

    return "";
}

static inline void mark_buffer_overflow(void)
{
    inst_boot_log_phdr->islogbuffull = 1;
    ibmemcpy(inst_boot_log_phdr->plogbuf, (uint8_t*)INST_BOOT_OVERFLOW_MSG, INST_BOOT_OVERFLOW_MSG_SIZE);
    inst_boot_log_phdr->plogbuf += INST_BOOT_OVERFLOW_MSG_SIZE;
    inst_boot_log_phdr->logsize += INST_BOOT_OVERFLOW_MSG_SIZE;
}

#define INST_BOOT_ABS_COUNT_STR_LEN     5
static inline void put_abs_count_in_log_buffer(uint64_t count)
{
    uint8_t i = 0;
    char *pcountstr = (char*)inst_boot_log_phdr->plogbuf;
    uint32_t spaceleft = INST_BOOT_LOG_LOGBUFFER_SIZE(inst_boot_log_phdr->bufsize) - inst_boot_log_phdr->logsize;
    uint32_t remainder = 0;
    count = (0ULL == count) ? get_cntpct_count() : count;
    count *= inst_boot_log_phdr->count_res;
    remainder = do_div(count, inst_boot_log_phdr->cntpctfreq[inst_boot_log_phdr->bootstage-1]);

    if(0 == pcountstr)
    {
        return;
    }

    if(spaceleft < INST_BOOT_ABS_COUNT_STR_LEN)
    {
        mark_buffer_overflow();
        return;
    }

    for(i = INST_BOOT_ABS_COUNT_STR_LEN; i > 0; i--)
    {
        if(0 != count)
        {
            pcountstr[i-1] = (char)do_div(count, 10) + '0';
        }
        else
        {
            pcountstr[i-1] = ' ';
        }
    }

    inst_boot_log_phdr->plogbuf += INST_BOOT_ABS_COUNT_STR_LEN;
    inst_boot_log_phdr->logsize += INST_BOOT_ABS_COUNT_STR_LEN;
    return;
}

#define INST_BOOT_DELTA_COUNT_STR_LEN     5
extern uint64_t inst_boot_previous_count;
static inline void put_delta_count_in_log_buffer(uint64_t count)
{
    uint8_t i = 0;
    char *pcountstr = (char*)inst_boot_log_phdr->plogbuf;
    uint32_t spaceleft = INST_BOOT_LOG_LOGBUFFER_SIZE(inst_boot_log_phdr->bufsize) - inst_boot_log_phdr->logsize;
    uint32_t remainder = 0;
    uint64_t display_count;
    int plus = 0;
    count = (0ULL == count) ? get_cntpct_count() : count;
    count *= inst_boot_log_phdr->count_res;
    remainder = do_div(count, inst_boot_log_phdr->cntpctfreq[inst_boot_log_phdr->bootstage-1]);
    display_count = count - inst_boot_previous_count;

    inst_boot_previous_count = count;

    if(0 == pcountstr)
    {
        return;
    }

    if(spaceleft < INST_BOOT_DELTA_COUNT_STR_LEN)
    {
        if (INST_BOOT_WRAP_ON_OVERFLOW) {
            // reset to start of log (base + header)
            inst_boot_log_phdr->plogbuf = inst_boot_log_phdr->pbufbase;
            inst_boot_log_phdr->plogbuf += INST_BOOT_LOG_HEADER_SIZE;
            // write OVERFLOW message at top
            ibmemcpy(inst_boot_log_phdr->plogbuf, (uint8_t*)INST_BOOT_OVERFLOW_MSG, INST_BOOT_OVERFLOW_MSG_SIZE);
            inst_boot_log_phdr->plogbuf += INST_BOOT_OVERFLOW_MSG_SIZE;
            inst_boot_log_phdr->logsize = INST_BOOT_OVERFLOW_MSG_SIZE;
            // reset pcountstr
            pcountstr = (char*)inst_boot_log_phdr->plogbuf;
        } else {
            mark_buffer_overflow();
            return;
        }
    }

    if (display_count == 0)
    {
        pcountstr[INST_BOOT_DELTA_COUNT_STR_LEN - 1] = '0';
        pcountstr[INST_BOOT_DELTA_COUNT_STR_LEN - 2] = '+';
        ibmemset(pcountstr, INST_BOOT_DELTA_COUNT_STR_LEN - 2, ' ');
    }
    else
    {
        for(i = INST_BOOT_DELTA_COUNT_STR_LEN; i > 0; i--)
        {
            if(0 != display_count)
            {
                pcountstr[i-1] = (char)do_div(display_count, 10) + '0';
            }
            else if (!plus)
            {
                pcountstr[i-1] = '+';
                plus = 1;
            }
            else
            {
                pcountstr[i-1] = ' ';
            }
        }
    }

    inst_boot_log_phdr->plogbuf += INST_BOOT_DELTA_COUNT_STR_LEN;
    inst_boot_log_phdr->logsize += INST_BOOT_DELTA_COUNT_STR_LEN;
    return;
}

static inline void put_spaces_in_log_buffer(int x)
{
    uint32_t spaceleft = INST_BOOT_LOG_LOGBUFFER_SIZE(inst_boot_log_phdr->bufsize) - inst_boot_log_phdr->logsize;

    if (spaceleft < x)
    {
        mark_buffer_overflow();
        return;
    }

    ibmemset(inst_boot_log_phdr->plogbuf, x, ' ');

    inst_boot_log_phdr->plogbuf += x;
    inst_boot_log_phdr->logsize += x;
}

static inline void put_msg_in_log_buffer(char* msg)
{
    uint32_t msglen = 0;
    char *ptr = 0;
    uint32_t spaceleft = INST_BOOT_LOG_LOGBUFFER_SIZE(inst_boot_log_phdr->bufsize) - inst_boot_log_phdr->logsize;

    for(ptr = msg; *ptr; ptr++)
    {
        msglen++;
    }

    if(0 == msglen)
    {
        return;
    }

    if(spaceleft < msglen)
    {
        if (INST_BOOT_WRAP_ON_OVERFLOW) {
            // reset to start of log (base + header)
            inst_boot_log_phdr->plogbuf = inst_boot_log_phdr->pbufbase;
            inst_boot_log_phdr->plogbuf += INST_BOOT_LOG_HEADER_SIZE;
            // write OVERFLOW message at top
            ibmemcpy(inst_boot_log_phdr->plogbuf, (uint8_t*)INST_BOOT_OVERFLOW_MSG, INST_BOOT_OVERFLOW_MSG_SIZE);
            inst_boot_log_phdr->plogbuf += INST_BOOT_OVERFLOW_MSG_SIZE;
            inst_boot_log_phdr->logsize = INST_BOOT_OVERFLOW_MSG_SIZE;
        } else {
            mark_buffer_overflow();
            return;
        }
    }

    ibmemcpy(inst_boot_log_phdr->plogbuf, (uint8_t*)msg, msglen);
    inst_boot_log_phdr->plogbuf += msglen;
    inst_boot_log_phdr->logsize += msglen;
    return;
}

static inline void put_count_and_msg_in_log_buffer(uint64_t count, char *msg_pre_tag, char* msg_tag, char* msg, int include_delta)
{
    if(0 == msg_tag)
    {
        return;
    }

    if(0 == msg)
    {
        return;
    }

    if(0 != inst_boot_log_phdr->islogbuffull)
    {
        return;
    }

    put_abs_count_in_log_buffer(count);
    put_msg_in_log_buffer(get_count_res_str());
    if (include_delta && (inst_boot_previous_count != 0))
    {
        put_delta_count_in_log_buffer(count);
    }
    else
    {
        count *= inst_boot_log_phdr->count_res;
        do_div(count, inst_boot_log_phdr->cntpctfreq[inst_boot_log_phdr->bootstage-1]);
        inst_boot_previous_count = count;
        put_spaces_in_log_buffer(INST_BOOT_DELTA_COUNT_STR_LEN);
    }
    put_msg_in_log_buffer(" ");
    put_msg_in_log_buffer(msg_pre_tag);
    put_msg_in_log_buffer(": ");
    put_msg_in_log_buffer(msg_tag);
    put_msg_in_log_buffer(" ");
    put_msg_in_log_buffer(msg);
    put_msg_in_log_buffer("\n");
}

#endif // INST_BOOT_LOG_LOCAL_H_