/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#define INST_BOOT_LOG_C

/*
    This file contains the parts of the inst_boot_log implementation that are shared by the kernel
    and the decompressor.
*/

#include "inst_boot_log_local.h"

uint64_t inst_boot_previous_count;

/* External interfaces implemented here */
int inst_boot_log_init(void* pbuf, unsigned int size, unsigned int count_res, char* msg)
{
    INST_BOOT_LOGGING_OVERHEAD_START;
    uint64_t count = 0ULL;
    unsigned long irq_flags;

    if(inst_boot_log_lock_initialized == 0)
    {
        /* Can't use the spin_lock_init() macro because it declares a static disallowed by the decomrpessor */
#ifdef CONFIG_DEBUG_SPINLOCK
        struct lock_class_key key;
        spinlock_check(&inst_boot_log_lock);
        __raw_spin_lock_init(&(inst_boot_log_lock.rlock), "inst_boot_log_lock", &key);
#else
        raw_spin_lock_init(&(inst_boot_log_lock.rlock));
#endif
        inst_boot_log_lock_initialized = 1;
    }

    spin_lock_irqsave(&inst_boot_log_lock, irq_flags);

    inst_boot_log_phdr = (inst_boot_log_hdr*)INST_BOOT_LOG_HEADER_ADDR((uint8_t*)pbuf);
    if(0 == (uint8_t*)pbuf)
    {
        return IB_NULL_PTR;
    }

    if(0 != inst_boot_log_phdr->initdone)
    {
        /* last stage bootloader didn't exit */
        inst_boot_log_phdr->initdone = 0;

        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return IB_PREV_BL_DIDNT_EXIT;
    }

    if (BOOT_STAGES <= inst_boot_log_phdr->bootstage)
    {
        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return IB_FAIL;
    }

    count = get_cntpct_count();
    if(INST_BOOT_MAGIC_NUMBER == inst_boot_log_phdr->magic_number)
    {
        inst_boot_log_phdr->bootstage++;
        inst_boot_log_phdr->oninitcntpctcount[inst_boot_log_phdr->bootstage-1] = count;
        inst_boot_log_phdr->cntpctfreq[inst_boot_log_phdr->bootstage-1] = get_cntpct_freq();
        inst_boot_log_phdr->count_res = count_res;
        inst_boot_log_phdr->plogbuf = (uint8_t*)pbuf + (inst_boot_log_phdr->plogbuf - inst_boot_log_phdr->pbufbase);
        inst_boot_log_phdr->pbufbase = (uint8_t*)pbuf;
    }
    else
    {
        if(INST_BOOT_LOG_MIN_LOGBUFFER_SIZE > size)
        {
            spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
            return IB_NOT_ENOUGH_LOG_MEM;
        }

        ibmemset((uint8_t*)inst_boot_log_phdr, INST_BOOT_LOG_HEADER_SIZE, 0);

        inst_boot_log_phdr->magic_number = INST_BOOT_MAGIC_NUMBER;
        inst_boot_log_phdr->logging_overhead = 0;
        inst_boot_log_phdr->pbufbase = (uint8_t*)pbuf;
        inst_boot_log_phdr->bufsize = size;
        inst_boot_log_phdr->plogbuf = (uint8_t*)INST_BOOT_LOG_LOGBUFFER_ADDR(inst_boot_log_phdr->pbufbase);
        inst_boot_log_phdr->logsize = 0;
        inst_boot_log_phdr->bootstage++;
        inst_boot_log_phdr->oninitcntpctcount[inst_boot_log_phdr->bootstage-1] = count;
        inst_boot_log_phdr->cntpctfreq[inst_boot_log_phdr->bootstage-1] = get_cntpct_freq();
        inst_boot_log_phdr->count_res = count_res;
        inst_boot_log_phdr->memtype = MEM_2;
        inst_boot_log_phdr->islogbuffull = 0;
    }

    inst_boot_log_phdr->initdone = 1;
    put_count_and_msg_in_log_buffer(count, INST_BOOT_LOGGING_KERNEL_PRE_TAG, "BS", msg, 0);
    INST_BOOT_LOGGING_OVERHEAD_END;

    spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
    return IB_SUCCESS;
}

void inst_boot_log_exit(char* msg)
{
    INST_BOOT_LOGGING_OVERHEAD_START;
    uint64_t count = get_cntpct_count();
    unsigned long irq_flags;
    spin_lock_irqsave(&inst_boot_log_lock, irq_flags);

    if(0 == inst_boot_log_phdr)
    {
        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return;
    }

    if(0 == inst_boot_log_phdr->initdone)
    {
        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return;
    }

    put_count_and_msg_in_log_buffer(count, INST_BOOT_LOGGING_KERNEL_PRE_TAG, "BE", msg, 1);
    INST_BOOT_LOGGING_OVERHEAD_END;

    if(0 != inst_boot_log_phdr)
    {
        inst_boot_log_phdr->initdone = 0;
        inst_boot_log_phdr = 0;
    }

    spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
    return;
}

void inst_boot_log_msg(char *msg_pre_tag, char* msg_tag, char* msg)
{
    INST_BOOT_LOGGING_OVERHEAD_START;
    unsigned long irq_flags;
    spin_lock_irqsave(&inst_boot_log_lock, irq_flags);

    if(0 == inst_boot_log_phdr)
    {
        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return;
    }

    if(0 == inst_boot_log_phdr->initdone)
    {
        spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
        return;
    }

    put_count_and_msg_in_log_buffer(0ULL, msg_pre_tag ? msg_pre_tag : INST_BOOT_LOGGING_KERNEL_PRE_TAG, msg_tag, msg, 1);
    INST_BOOT_LOGGING_OVERHEAD_END;
    spin_unlock_irqrestore(&inst_boot_log_lock, irq_flags);
    return;
}

uint64_t inst_boot_get_cnt(void)
{
    return get_cntpct_count();
}

uint32_t inst_boot_get_freq(void)
{
    return get_cntpct_freq();
}

