/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/err.h>
#include <linux/regulator/consumer.h>
#include <linux/miscdevice.h>
#include <linux/device.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/delay.h>
#include <linux/uaccess.h>
#include <linux/lytmcu.h>

struct lytmcu_device
{
    struct miscdevice dev;
    struct platform_driver lytmcu_platform_driver;
    struct file_operations fops;
    struct regulator *vdd;
    int reset_gpio;
    u32 reset_gpio_flags;
    int bootconfig_gpio;
    u32 bootconfig_gpio_flags;
};

static struct lytmcu_device lytmcu;

static long lytmcu_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    struct lytmcu_data data;

    if (copy_from_user(&data, (void __user *)arg, sizeof(struct lytmcu_data)))
    {
        return -EFAULT;
    }

    switch (cmd)
    {
        case LYTMCU_IOC_RESET:
        {
            gpio_set_value(lytmcu.reset_gpio, data.value);
            break;
        }
        case LYTMCU_IOC_POWER:
        {
            if (0 == data.value)
            {
                regulator_disable(lytmcu.vdd);
            }
            else
            {
                regulator_enable(lytmcu.vdd);
            }
            break;
        }
        case LYTMCU_IOC_BOOTMODE:
        {
            gpio_set_value(lytmcu.bootconfig_gpio, data.value);
            break;
        }
        default:
        {
            pr_err("lytmcu_ioctl : unknown cmd %x\n", cmd);
            break;
        }
    }

    return 0;
}

static int lytmcu_probe(struct platform_device *pdev)
{
    int error = 0;
    lytmcu.vdd = regulator_get(&pdev->dev, "vdd");
    if (IS_ERR(lytmcu.vdd)) {
        error = PTR_ERR(lytmcu.vdd);
        dev_err(&pdev->dev, "Regulator get failed vdd error=%d\n", error);
    }
    else
    {
        lytmcu.reset_gpio = of_get_named_gpio_flags(pdev->dev.of_node, "lyt,reset-gpio", 0, &lytmcu.reset_gpio_flags);
        lytmcu.bootconfig_gpio = of_get_named_gpio_flags(pdev->dev.of_node, "lyt,bootconfig-gpio", 0, &lytmcu.bootconfig_gpio_flags);

        error = gpio_request(lytmcu.reset_gpio, "lyt_mcu_reset_gpio");
        if (error)
        {
            dev_err(&pdev->dev, "reset gpio request failed error=%d\n", error);
            goto exit;
        }

        error = gpio_request(lytmcu.bootconfig_gpio, "lyt_mcu_bootconfig_gpio");
        if (error)
        {
            dev_err(&pdev->dev, "bootconfig gpio request failed error=%d\n", error);
            gpio_free(lytmcu.reset_gpio);
            goto exit;
        }
    }

exit:
    return error;
}

static struct of_device_id lytmcu_of_match[] = {
    { .compatible = "lyt,mcu" },
    { },
};

static struct lytmcu_device lytmcu = {
    .dev = {
        .minor = MISC_DYNAMIC_MINOR,
        .name = "lytmcumisc",
        .fops = &lytmcu.fops,
        .mode = 0666
    },
    .lytmcu_platform_driver = {
        .probe      = lytmcu_probe,

        .driver     = {
            .name   = "lytmcuplatform",
            .owner  = THIS_MODULE,
            .of_match_table = lytmcu_of_match,
        },
    },
    .fops = {
        .owner = THIS_MODULE,
        .unlocked_ioctl = lytmcu_ioctl,
        .compat_ioctl = lytmcu_ioctl
    },
};

static int __init lytmcu_init(void)
{
    int ret = 0;

    ret = misc_register(&lytmcu.dev);
    if(unlikely(ret))
    {
        pr_err("lytmcu: failed to register misc device!\n");
        goto misc_register_fail;
    }

    ret = platform_driver_register(&lytmcu.lytmcu_platform_driver);
    if(unlikely(ret))
    {
        pr_err("lytmcu: failed to register platform device!\n");
        goto platform_register_fail;
    }

    return 0;

platform_register_fail:
    misc_deregister(&lytmcu.dev);
misc_register_fail:
    return ret;
}
late_initcall(lytmcu_init);

static void __exit lytmcu_exit(void)
{
    gpio_free(lytmcu.reset_gpio);
    gpio_free(lytmcu.bootconfig_gpio);
    platform_driver_unregister(&lytmcu.lytmcu_platform_driver);
    misc_deregister(&lytmcu.dev);
}
module_exit(lytmcu_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LYT MCU driver");
MODULE_VERSION("1.0");
MODULE_AUTHOR("Ashish Singh");
