/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/security.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/uaccess.h>
#include <linux/personality.h>
#include <linux/bitops.h>
#include <linux/mutex.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/inst_boot.h>
#include <linux/inst_boot_log.h>
#include <linux/inst_boot_log_ddr_info.h>


#define INST_BOOT_MEMORY_BASE      INST_BOOT_LOG_DDR_MEM_BASE
#define INST_BOOT_MEMORY_SIZE      INST_BOOT_LOG_DDR_MEM_SIZE

struct inst_boot_device
{
    struct miscdevice dev;
    struct file_operations fops;
    struct mutex lock;
    struct resource *res;
    void *memptr;
};

static struct inst_boot_device inst_boot;

static inline int32_t inst_boot_validate_address(uint32_t addr, uint32_t datasize)
{
    int32_t ret = 0;

    if(addr < INST_BOOT_MEMORY_BASE ||
       addr > (INST_BOOT_MEMORY_BASE + INST_BOOT_MEMORY_SIZE - datasize))
        ret = -1;

    return ret;
}

static inline uint32_t inst_boot_translate_address(uint32_t phyaddr)
{
    return (phyaddr - INST_BOOT_MEMORY_BASE) + (uint32_t)inst_boot.memptr;
}

static long inst_boot_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
    switch (cmd)
    {
        case INST_BOOT_IOC_MEMORY_TEST: /* _IOR */
        {
            int32_t testret = 0;
            uint32_t i = 0;
            uint32_t cnt = 0;
            uint32_t addr = 0;
            uint32_t size = INST_BOOT_MEMORY_SIZE>>2;

            // Initital read.
            for(i = 0, cnt = 0; i < size; i++)
            {
                addr = inst_boot_translate_address(INST_BOOT_MEMORY_BASE + (i<<2));
                if(i == ioread32(addr))
                    cnt++;
            }

            printk(KERN_INFO "inst_boot: Before Write Test\n");
            if (cnt == size)
            {
                printk(KERN_INFO "inst_boot: Surprise! Memory state not as "
                                 "expected. This should not happen.\n");
            }
            else
            {
                printk(KERN_INFO "inst_boot: Memory state as expected before "
                                 "running the write test.\n");
            }

            // Write Test.
            for(i = 0; i < size; i++)
            {
                addr = inst_boot_translate_address(INST_BOOT_MEMORY_BASE + (i<<2));
                iowrite32(i, addr);
            }

            // Verifying write.
            for(i = 0, cnt = 0; i < size; i++)
            {
                addr = inst_boot_translate_address(INST_BOOT_MEMORY_BASE + (i<<2));
                if(i == ioread32(addr))
                    cnt++;
            }

            printk(KERN_INFO "inst_boot: After Write Test\n");
            if (cnt == size)
            {
                printk(KERN_INFO "inst_boot: Test Pass.\n");
            }
            else
            {
                testret = -1;
                printk(KERN_INFO "inst_boot: Test Fail.\n");
            }

            // Write all zeros.
            for(i = 0; i < size; i++)
            {
                addr = inst_boot_translate_address(INST_BOOT_MEMORY_BASE + (i<<2));
                iowrite32(0, addr);
            }

            if(copy_to_user((void __user *)arg, &testret, sizeof(int)))
                return -EFAULT;

            break;
        }

        case INST_BOOT_IOC_MEMORY_RD32: /* _IOWR  */
        {
            struct inst_boot_rwdata rwdata = {0};
            uint32_t addr = 0;

            if(copy_from_user(&rwdata, (void __user *)arg, sizeof(struct inst_boot_rwdata)))
                return -EFAULT;

            if(inst_boot_validate_address(rwdata.addr, 4) == -1)
            {
                rwdata.ret = -1;
            }
            else
            {
                addr = inst_boot_translate_address(rwdata.addr);
                rwdata.data = ioread32(addr);
            }

            if(copy_to_user((void __user *)arg, &rwdata, sizeof(struct inst_boot_rwdata)))
                return -EFAULT;

            break;
        }

        case INST_BOOT_IOC_MEMORY_RD_CHAR_ARRAY: /* _IOWR  */
        {
            uint32_t addr = 0;
            uint32_t valid_reads = 0;
            int32_t ret = 0;
            struct inst_boot_rd_char_array *padata = NULL;

            padata = kmalloc(sizeof(*padata), GFP_KERNEL);
            if(padata == NULL)
                return -EFAULT;

            padata->ret = 0;

            if(copy_from_user(padata, (void __user *)arg, sizeof(struct inst_boot_rd_char_array)))
            {
                ret = -EFAULT;
                goto array_read_exit;
            }

            if(inst_boot_validate_address(padata->addr, 1) == -1)
            {
                padata->ret = -1;
            }
            else
            {
                /* count valid reads */
                for(valid_reads = 0; (inst_boot_validate_address(padata->addr + valid_reads, 1) != -1) &&
                                     (valid_reads < padata->toread) &&
                                     (valid_reads < INST_BOOT_ARRAY_SIZE); valid_reads++);

                //printk(KERN_INFO "inst_boot: valid reads=%d [%x].\n", valid_reads, padata->addr);

                for(padata->read = 0; padata->read < valid_reads; padata->read++)
                {
                    addr = inst_boot_translate_address(padata->addr + padata->read);
                    padata->data[padata->read] = ioread8(addr);
                }
            }

            if(copy_to_user((void __user *)arg, padata, sizeof(struct inst_boot_rd_char_array)))
            {
                ret = -EFAULT;
                goto array_read_exit;
            }

array_read_exit:
            if(padata)
                kfree(padata);

            if(ret != 0)
                return -EFAULT;

            break;
        }

        case INST_BOOT_IOC_MEMORY_WR32: /* _IOWR  */
        {
            struct inst_boot_rwdata rwdata = {0};
            uint32_t addr = 0;

            if(copy_from_user(&rwdata, (void __user *)arg, sizeof(struct inst_boot_rwdata)))
                return -EFAULT;

            if(inst_boot_validate_address(rwdata.addr, 4) == -1)
            {
                rwdata.ret = -1;
            }
            else
            {
                addr = inst_boot_translate_address(rwdata.addr);
                iowrite32(rwdata.data, addr);
            }

            if(copy_to_user((void __user *)arg, &rwdata, sizeof(struct inst_boot_rwdata)))
                return -EFAULT;

            break;
        }

        case INST_BOOT_IOC_MEMORY_INFO: /* _IOR  */
        {
            struct inst_boot_info info = {0};

            info.baseaddr = INST_BOOT_LOG_DDR_MEM_BASE;
            info.size = INST_BOOT_LOG_DDR_MEM_SIZE;
            info.ret = 0;

            if(copy_to_user((void __user *)arg, &info, sizeof(struct inst_boot_info)))
                return -EFAULT;

            break;
        }

        default:
        {
            printk(KERN_ERR "inst_boot_ioctl : unknown cmd %x\n", cmd);
            break;
        }
    }
    return 0;
}

ssize_t inst_boot_read(struct file *file, char __user *ubuf, size_t size, loff_t *ppos)
{
    void *addr= 0;
    size_t available_size = 0;
    if(inst_boot_log_get_info(0, &addr, 0, &available_size))
    {
        return simple_read_from_buffer(ubuf, size, ppos, addr, available_size);
    }
    return -EFAULT;
}

ssize_t inst_boot_write(struct file *file, const char __user *ubuf, size_t size, loff_t *ppos)
{
    char tag[64];
    char *buf, *first_newline;
    buf = kzalloc(size, GFP_KERNEL);

	if (!buf)
    {
        return -ENOMEM;
    }

    if(copy_from_user(buf, (void __user *)ubuf, size))
    {
        kfree(buf);
        return -EFAULT;
    }

    // We want to just grab the first line of the write if it's a multi-line write
    // We also want to eat the newline because inst_boot formats with newlines
    first_newline = strchr(buf, '\n');
    if(first_newline)
    {
        *first_newline = '\0';
    }

    snprintf(tag, sizeof(tag), "(%d)", current->pid);


    inst_boot_log_msg("USER", tag, buf);

    kfree(buf);

    return size;
}

static struct inst_boot_device inst_boot = {
    .dev = {
        .minor = MISC_DYNAMIC_MINOR,
        .name = "inst_boot",
        .fops = &inst_boot.fops,
        .mode = 0666 },
    .fops = {
        .owner = THIS_MODULE,
        .unlocked_ioctl = inst_boot_ioctl,
        .compat_ioctl = inst_boot_ioctl,
        .read = inst_boot_read,
        .write = inst_boot_write },
};


static int __init inst_boot_init(void)
{
    int ret;

    mutex_init(&inst_boot.lock);

    ret = misc_register(&inst_boot.dev);
    if(unlikely(ret))
    {
        printk(KERN_ERR "inst_boot: failed to register misc device!\n");
        goto misc_register_fail;
    }

    inst_boot.res = request_mem_region(INST_BOOT_MEMORY_BASE,
                                       INST_BOOT_MEMORY_SIZE, "INSTBOOT");
    if(NULL == inst_boot.res)
    {
        printk(KERN_ERR "inst_boot: failed to request memory region! [%x, %x]\n",
               INST_BOOT_MEMORY_BASE, INST_BOOT_MEMORY_SIZE);
        ret = -ENOMEM;
        goto request_mem_region_fail;
    }

    inst_boot.memptr = ioremap(INST_BOOT_MEMORY_BASE, INST_BOOT_MEMORY_SIZE);
    if(NULL == inst_boot.memptr)
    {
        printk(KERN_ERR "inst_boot: failed ioremap!\n");
        ret = -ENOMEM;
        goto ioremap_fail;
    }

    printk(KERN_INFO "inst_boot: initialized [%d, %d] [%p, %p]\n",
           MAJOR(inst_boot.dev.this_device->devt),
           MINOR(inst_boot.dev.this_device->devt),
           inst_boot.res,
           inst_boot.memptr);

    return ret;

ioremap_fail:
    release_mem_region(INST_BOOT_MEMORY_BASE, INST_BOOT_MEMORY_SIZE);
request_mem_region_fail:
    misc_deregister(&inst_boot.dev);
misc_register_fail:
    return ret;
}

static void __exit inst_boot_exit(void)
{
    int ret;

    iounmap(inst_boot.memptr);
    release_mem_region(INST_BOOT_MEMORY_BASE, INST_BOOT_MEMORY_SIZE);

    ret = misc_deregister(&inst_boot.dev);
    if (unlikely(ret))
    {
        printk(KERN_ERR "inst_boot: failed to unregister misc device!\n");
    }

    printk(KERN_INFO "inst_boot: unloaded\n");
}

module_init(inst_boot_init);
module_exit(inst_boot_exit);

MODULE_LICENSE("GPL");
