/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/security.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/uaccess.h>
#include <linux/personality.h>
#include <linux/bitops.h>
#include <linux/mutex.h>
#include <linux/fd_porter.h>

struct fd_porter_device
{
    struct miscdevice dev;
    struct file_operations fops;
    struct mutex lock;
};


struct fd_porter_file_wrapper
{
    struct file *file;
    int orig_fd;
    pid_t orig_pid;
};


int fd_porter_import(struct fd_porter_device *device, struct fd_porter_data *data)
{
    struct fd_porter_file_wrapper *porter_file;
    struct file *file;
    int fd;

    if (!device || !data)
    {
        return -EINVAL;
    }

    mutex_lock(&device->lock);

    porter_file = data->handle;
    if(!porter_file)
    {
        printk(KERN_ERR "fd_porter_import : porter_file is null\n");
        mutex_unlock(&device->lock);
        return -EINVAL;
    }

    file = porter_file->file;
    if(!file)
    {
        printk(KERN_ERR "fd_porter_import : porter_file->file is null\n");
        mutex_unlock(&device->lock);
        return -ENOENT;
    }

    fd = get_unused_fd_flags(O_CLOEXEC);
    if(fd < 0)
    {
        printk(KERN_ERR "fd_porter_import : get_unused_fd_flags failed\n");
        mutex_unlock(&device->lock);
        return fd;
    }

    // Increment the ref count. Matching decrement is when user calls close() on the fd.
    get_file(file);

    fd_install(fd, file);

    data->fd = fd;
    //printk(KERN_INFO "fd_porter_import: %p -> %x [count %d]\n", data->handle, data->fd, (int)file_count(file));
    mutex_unlock(&device->lock);
    return 0;
}

EXPORT_SYMBOL(fd_porter_import);

int fd_porter_export(struct fd_porter_device *device, struct fd_porter_data *data)
{
    struct fd_porter_file_wrapper *porter_file;
    struct file *file;

    if (!device || !data)
    {
        return -EINVAL;
    }

    mutex_lock(&device->lock);

    // Call to fget increments the ref count. Matching decrement is in fd_porter_delete.
    file = fget(data->fd);

    if(!file)
    {
        printk(KERN_ERR "fd_porter_export : file is null\n");
        mutex_unlock(&device->lock);
        return -ENOENT;
    }

    porter_file = kzalloc(sizeof(struct fd_porter_file_wrapper), GFP_KERNEL);
    if(!porter_file)
    {
        printk(KERN_ERR "fd_porter_export : porter_file couldn't be allocated\n");
        mutex_unlock(&device->lock);
        return -ENOMEM;
    }

    porter_file->file = file;
    porter_file->orig_fd = data->fd;
    porter_file->orig_pid = current->pid;

    data->handle = (void*)porter_file;
    //printk(KERN_INFO "fd_porter_export: %x -> %p [count %d]\n", data->fd, data->handle, (int)file_count(file));
    mutex_unlock(&device->lock);
    return 0;
}

EXPORT_SYMBOL(fd_porter_export);


int fd_porter_delete(struct fd_porter_device *device, struct fd_porter_data *data)
{
    struct fd_porter_file_wrapper *porter_file;

    if (!device || !data)
    {
        return -EINVAL;
    }

    mutex_lock(&device->lock);

    porter_file = data->handle;
    if(!porter_file || !porter_file->file)
    {
        printk(KERN_ERR "fd_porter_delete : %s is null\n", porter_file ? "porter_file" : "porter_file->file");
        mutex_unlock(&device->lock);
        return -EINVAL;
    }

    // Decrement the ref count. Matching increment is in fd_porter_export.
    fput(porter_file->file);

    kfree(porter_file);

    mutex_unlock(&device->lock);
    return 0;
}

EXPORT_SYMBOL(fd_porter_delete);

static struct fd_porter_device fd_porter;

static long fd_porter_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
    struct fd_porter_device *device = &fd_porter;
    long ret = -EINVAL;

    switch (cmd)
    {
        case FD_PORTER_IOC_EXPORT:
        {
            struct fd_porter_data data;

            if(copy_from_user(&data, (void __user *)arg, sizeof(struct fd_porter_data)))
                return -EFAULT;

            ret = fd_porter_export(device, &data);
            if (ret < 0)
                return ret;

            if(copy_to_user((void __user *)arg, &data, sizeof(struct fd_porter_data)))
                return -EFAULT;

            break;
        }
        case FD_PORTER_IOC_IMPORT:
        {
            struct fd_porter_data data;

            if(copy_from_user(&data, (void __user *)arg, sizeof(struct fd_porter_data)))
                return -EFAULT;

            ret = fd_porter_import(device, &data);
            if (ret < 0)
                return ret;

            if(copy_to_user((void __user *)arg, &data, sizeof(struct fd_porter_data)))
                return -EFAULT;

            break;
        }
        case FD_PORTER_IOC_DELETE:
        {
            struct fd_porter_data data;

            if(copy_from_user(&data, (void __user *)arg, sizeof(struct fd_porter_data)))
                return -EFAULT;

            ret = fd_porter_delete(device, &data);
            if (ret < 0)
                return ret;

            if(copy_to_user((void __user *)arg, &data, sizeof(struct fd_porter_data)))
                return -EFAULT;

            break;
        }
        default:
        {
            printk(KERN_ERR "fd_porter_ioctl : unknown cmd %x\n", cmd);
            break;
        }
    }
    return ret;
}

static struct fd_porter_device fd_porter = {
    .dev = {
        .minor = MISC_DYNAMIC_MINOR,
        .name = "fd_porter",
        .fops = &fd_porter.fops,
        .mode = 0666 },
    .fops = {
        .owner = THIS_MODULE,
        .unlocked_ioctl = fd_porter_ioctl,
        .compat_ioctl = fd_porter_ioctl },
};


static int __init fd_porter_init(void)
{
    int ret;

    mutex_init(&fd_porter.lock);

    ret = misc_register(&fd_porter.dev);
    if(unlikely(ret))
    {
        printk(KERN_ERR "fd_porter: failed to register misc device!\n");
        return ret;
    }

    printk(KERN_INFO "fd_porter: initialized\n");

    return 0;
}

static void __exit fd_porter_exit(void)
{
    int ret;

    ret = misc_deregister(&fd_porter.dev);
    if (unlikely(ret))
    {
        printk(KERN_ERR "fd_porter: failed to unregister misc device!\n");
    }

    printk(KERN_INFO "fd_porter: unloaded\n");
}

module_init(fd_porter_init);
module_exit(fd_porter_exit);

MODULE_LICENSE("GPL");
