/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "msm_sensor.h"
#include "camera.h"



#undef CDBG

/*#define CONFIG_MSMB_CAMERA_DEBUG*/

#ifdef CONFIG_MSMB_CAMERA_DEBUG
#define CDBG(fmt, args...) pr_err(fmt, ##args)
#else
#define CDBG(fmt, args...) do { } while (0)
#endif



DEFINE_MSM_MUTEX(virt_sens_mut);

#define VIRT_SENS_NODE_POSITION (5)

static struct msm_sensor_ctrl_t *get_sctrl(struct v4l2_subdev *sd)
{
	return container_of(container_of(sd, struct msm_sd_subdev, sd),
		struct msm_sensor_ctrl_t, msm_sd);
}

long msm_virt_sensor_subdev_ioctl(struct v4l2_subdev *sd,
			unsigned int cmd, void *arg)
{
	struct msm_sensor_ctrl_t *s_ctrl = get_sctrl(sd);
	void __user *argp = (void __user *)arg;

	switch (cmd) {
	case VIDIOC_MSM_SENSOR_CFG:
		if (s_ctrl->func_tbl->sensor_config)
			return s_ctrl->func_tbl->sensor_config(s_ctrl, argp);
		else
			CDBG("%s(%d): sensor_config == NULL\n",
					__func__, __LINE__);
		return 0;
	case VIDIOC_MSM_SENSOR_GET_AF_STATUS:
		return -ENOIOCTLCMD;
	case VIDIOC_MSM_SENSOR_RELEASE:
		return 0;
	case MSM_SD_SHUTDOWN:
		return 0;
	default:
		return -ENOIOCTLCMD;
	}
}

static struct msm_sensor_ctrl_t virt_sens_s_ctrl;
static struct v4l2_subdev_core_ops virt_sens_subdev_core_ops = {
		.ioctl = msm_virt_sensor_subdev_ioctl,
};
static struct v4l2_subdev_video_ops virt_sens_subdev_video_ops;

static struct v4l2_subdev_ops virt_sens_subdev_ops = {
	.core = &virt_sens_subdev_core_ops,
	.video  = &virt_sens_subdev_video_ops,
};

static struct v4l2_subdev_info virt_sens_subdev_info[] = {
	{
		.code   = V4L2_MBUS_FMT_SBGGR10_1X10,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.fmt    = 1,
		.order    = 0,
	},
};

static const struct of_device_id virt_sens_dt_match[] = {
	{.compatible = "qcom,virt_sens", .data = &virt_sens_s_ctrl},
	{}
};

MODULE_DEVICE_TABLE(of, virt_sens_dt_match);

static struct platform_driver virt_sens_platform_driver = {
	.driver = {
		.name = "qcom,virt_sens",
		.owner = THIS_MODULE,
		.of_match_table = virt_sens_dt_match,
	},
};

static int32_t virt_sens_priv_get_dt_data(struct device_node *of_node,
	struct msm_sensor_ctrl_t *s_ctrl)
{
	int32_t rc = 0;
	struct msm_camera_sensor_board_info *sensordata = NULL;

	s_ctrl->sensordata = kzalloc(sizeof(
		struct msm_camera_sensor_board_info),
		GFP_KERNEL);
	if (!s_ctrl->sensordata) {
		pr_err("%s failed %d\n", __func__, __LINE__);
		return -ENOMEM;
	}

	sensordata = s_ctrl->sensordata;

	sensordata->sensor_init_params = kzalloc(sizeof(
		struct msm_sensor_init_params), GFP_KERNEL);
	if (!sensordata->sensor_init_params) {
		pr_err("%s failed %d\n", __func__, __LINE__);
		return -ENOMEM;
	}

	rc = of_property_read_string(of_node, "qcom,sensor-name",
		&sensordata->sensor_name);
	CDBG("%s qcom,sensor-name %s, rc %d\n", __func__,
		sensordata->sensor_name, rc);
	if (rc < 0) {
		pr_err("%s failed %d\n", __func__, __LINE__);
		goto ERROR1;
	}

	rc = of_property_read_u32(of_node, "qcom,sensor-mode",
		&sensordata->sensor_init_params->modes_supported);
	CDBG("%s qcom,sensor-mode %d, rc %d\n", __func__,
		sensordata->sensor_init_params->modes_supported, rc);
	if (rc < 0) {
		pr_err("%s failed %d\n", __func__, __LINE__);
		goto ERROR1;
	}

	rc = of_property_read_u32(of_node, "qcom,sensor-position",
		&sensordata->sensor_init_params->position);
	CDBG("%s qcom,sensor-position %d, rc %d\n", __func__,
		sensordata->sensor_init_params->position, rc);
	if (rc < 0) {
		pr_err("%s failed %d\n", __func__, __LINE__);
		goto ERROR1;
	}

	rc = of_property_read_u32(of_node, "qcom,mount-angle",
		&sensordata->sensor_init_params->sensor_mount_angle);
	CDBG("%s qcom,mount-angle %d, rc %d\n", __func__,
		sensordata->sensor_init_params->sensor_mount_angle, rc);
	if (rc < 0) {
		/* Set default mount angle */
		sensordata->sensor_init_params->sensor_mount_angle = 0;
		rc = 0;
	}

	return rc;

ERROR1:
	kfree(s_ctrl->sensordata);

	return rc;
}

int32_t msm_virt_sensor_config(struct msm_sensor_ctrl_t *s_ctrl,
				void __user *argp)
{
	struct sensorb_cfg_data *cdata = (struct sensorb_cfg_data *)argp;
	long rc = 0;
	int32_t i = 0;
	mutex_lock(s_ctrl->msm_sensor_mutex);
	CDBG("%s:%d %s cfgtype = %d\n", __func__, __LINE__,
		s_ctrl->sensordata->sensor_name, cdata->cfgtype);

	switch (cdata->cfgtype) {
	case CFG_GET_SENSOR_INFO:
		memcpy(cdata->cfg.sensor_info.sensor_name,
			s_ctrl->sensordata->sensor_name,
			sizeof(cdata->cfg.sensor_info.sensor_name));
		cdata->cfg.sensor_info.session_id =
			s_ctrl->sensordata->num_vreg;
		for (i = 0; i < SUB_MODULE_MAX; i++)
			cdata->cfg.sensor_info.subdev_id[i] = -1;

		CDBG("%s:%d sensor name %s\n", __func__, __LINE__,
			cdata->cfg.sensor_info.sensor_name);
		CDBG("%s:%d session id %d\n", __func__, __LINE__,
			cdata->cfg.sensor_info.session_id);
		for (i = 0; i < SUB_MODULE_MAX; i++)
			CDBG("%s:%d subdev_id[%d] %d\n", __func__, __LINE__, i,
				cdata->cfg.sensor_info.subdev_id[i]);

		break;
	case CFG_GET_SENSOR_INIT_PARAMS:
		cdata->cfg.sensor_init_params =
			*s_ctrl->sensordata->sensor_init_params;
		CDBG("%s:%d init params mode %d pos %d mount %d\n", __func__,
			__LINE__,
			cdata->cfg.sensor_init_params.modes_supported,
			cdata->cfg.sensor_init_params.position,
			cdata->cfg.sensor_init_params.sensor_mount_angle);
		break;

	case CFG_SET_SLAVE_INFO:
	case CFG_WRITE_I2C_ARRAY:
	case CFG_SLAVE_READ_I2C:
	case CFG_SLAVE_WRITE_I2C_ARRAY:
	case CFG_WRITE_I2C_SEQ_ARRAY:
	case CFG_POWER_UP:
	case CFG_POWER_DOWN:
	case CFG_SET_STOP_STREAM_SETTING:
		break;
	default:
		rc = -EFAULT;
		break;
	}

	mutex_unlock(s_ctrl->msm_sensor_mutex);

	return rc;
}

int virt_sensor_power_up(struct msm_sensor_ctrl_t * s_ctrl)
{
	s_ctrl = s_ctrl;
	CDBG("%s name %s\n", __func__, s_ctrl->sensordata->sensor_name);
	return 0;
}

int virt_sensor_power_down(struct msm_sensor_ctrl_t * s_ctrl)
{
	s_ctrl = s_ctrl;
	CDBG("%s name %s\n", __func__, s_ctrl->sensordata->sensor_name);
	return 0;
}


static struct msm_sensor_fn_t msm_virt_sensor_func_tbl = {
	.sensor_config = msm_virt_sensor_config,
	.sensor_power_up = virt_sensor_power_up,
	.sensor_power_down = virt_sensor_power_down,
	.sensor_match_id = virt_sensor_power_down,
};

static int32_t virt_sens_priv_platform_probe(struct platform_device *pdev,
	void *data)
{
	int32_t rc = 0;
	struct msm_sensor_ctrl_t *s_ctrl =
		(struct msm_sensor_ctrl_t *)data;
	uint32_t session_id;
	s_ctrl->pdev = pdev;
	s_ctrl->dev = &pdev->dev;
	CDBG("%s called data %p\n", __func__, data);
	CDBG("%s pdev name %s\n", __func__, pdev->id_entry->name);
	if (pdev->dev.of_node) {
		rc = virt_sens_priv_get_dt_data(pdev->dev.of_node, s_ctrl);
		if (rc < 0) {
			pr_err("%s failed line %d\n", __func__, __LINE__);
			return rc;
		}
	}

	if (!s_ctrl->func_tbl) {
		pr_info("%s %s Func_tbl\n", __func__,
			s_ctrl->sensordata->sensor_name);
		s_ctrl->func_tbl = &msm_virt_sensor_func_tbl;
	}

	if (!s_ctrl->sensor_v4l2_subdev_ops)
		s_ctrl->sensor_v4l2_subdev_ops = &virt_sens_subdev_ops;

	CDBG("%s %s probe succeeded\n", __func__,
		s_ctrl->sensordata->sensor_name);
	v4l2_subdev_init(&s_ctrl->msm_sd.sd,
		s_ctrl->sensor_v4l2_subdev_ops);
	snprintf(s_ctrl->msm_sd.sd.name,
		sizeof(s_ctrl->msm_sd.sd.name), "%s",
		s_ctrl->sensordata->sensor_name);
	v4l2_set_subdevdata(&s_ctrl->msm_sd.sd, pdev);
	s_ctrl->msm_sd.sd.flags |= V4L2_SUBDEV_FL_HAS_DEVNODE;
	media_entity_init(&s_ctrl->msm_sd.sd.entity, 0, NULL, 0);
	s_ctrl->msm_sd.sd.entity.type = MEDIA_ENT_T_V4L2_SUBDEV;
	s_ctrl->msm_sd.sd.entity.group_id = MSM_CAMERA_SUBDEV_SENSOR;
	s_ctrl->msm_sd.sd.entity.name =
		s_ctrl->msm_sd.sd.name;

	rc = camera_init_v4l2(&s_ctrl->pdev->dev, VIRT_SENS_NODE_POSITION,
		&session_id);
	s_ctrl->sensordata->num_vreg = (int)session_id;
	CDBG("%s rc %d session_id %d\n", __func__, rc, session_id);
	s_ctrl->msm_sd.close_seq = MSM_SD_CLOSE_2ND_CATEGORY | 0x3;
	msm_sd_register(&s_ctrl->msm_sd);
	CDBG("%s:%d\n", __func__, __LINE__);

	return rc;
}

static int32_t virt_sens_platform_probe(struct platform_device *pdev)
{
	int32_t rc = 0;
	const struct of_device_id *match;
	match = of_match_device(virt_sens_dt_match, &pdev->dev);
	rc = virt_sens_priv_platform_probe(pdev, match->data);
	return rc;
}

static int __init virt_sens_init_module(void)
{
	int32_t rc;
	pr_info("%s:%d\n", __func__, __LINE__);
	rc = platform_driver_probe(&virt_sens_platform_driver,
		virt_sens_platform_probe);
	if (!rc)
		return rc;
	return 0;
}

static void __exit virt_sens_exit_module(void)
{
	CDBG("%s:%d\n", __func__, __LINE__);
	if (virt_sens_s_ctrl.pdev) {
		msm_sensor_free_sensor_data(&virt_sens_s_ctrl);
		platform_driver_unregister(&virt_sens_platform_driver);
	}
	return;
}

static struct msm_sensor_ctrl_t virt_sens_s_ctrl = {
	.msm_sensor_mutex = &virt_sens_mut,
	.sensor_v4l2_subdev_info = virt_sens_subdev_info,
	.sensor_v4l2_subdev_info_size = ARRAY_SIZE(virt_sens_subdev_info),
};

module_init(virt_sens_init_module);
module_exit(virt_sens_exit_module);
MODULE_DESCRIPTION("virt_sens");
MODULE_LICENSE("GPL v2");
