/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "msm_sensor.h"
#define T8ES9_SENSOR_NAME "t8es9"
DEFINE_MSM_MUTEX(t8es9_mut);

static long t8es9_ioctl(struct file *file, unsigned int cmd, unsigned long arg);
static struct t8es9_device_t {
    struct miscdevice dev;
    struct file_operations fops;
} t8es9_device = {
    .dev = {
        .minor = MISC_DYNAMIC_MINOR,
        .name = "t8es9",
        .fops = &t8es9_device.fops,
        .mode = 0666 },
    .fops = {
        .owner = THIS_MODULE,
        .unlocked_ioctl = t8es9_ioctl,
        .compat_ioctl = t8es9_ioctl },
};

static struct msm_sensor_ctrl_t t8es9_s_ctrl;

static struct msm_sensor_power_setting t8es9_power_setting[] = {
    {
        .seq_type = SENSOR_VREG,
        .seq_val = CAM_VDIG,
        .config_val = 0,
        .delay = 0,
    },
    {
        .seq_type = SENSOR_VREG,
        .seq_val = CAM_VANA,
        .config_val = 0,
        .delay = 0,
    },
    {
        .seq_type = SENSOR_VREG,
        .seq_val = CAM_VIO,
        .config_val = 0,
        .delay = 0,
    },
    {
        .seq_type = SENSOR_VREG,
        .seq_val = CAM_VAF,
        .config_val = 0,
        .delay = 0,
    },
    {
        .seq_type = SENSOR_GPIO,
        .seq_val = SENSOR_GPIO_RESET,
        .config_val = GPIO_OUT_HIGH,
        .delay = 30,
    },
    {
        .seq_type = SENSOR_GPIO,
        .seq_val = SENSOR_GPIO_STANDBY,
        .config_val = GPIO_OUT_HIGH,
        .delay = 30,
    },
    {
        .seq_type = SENSOR_CLK,
        .seq_val = SENSOR_CAM_MCLK,
        .config_val = 0,
        .delay = 1,
    },
    {
        .seq_type = SENSOR_I2C_MUX,
        .seq_val = 0,
        .config_val = 0,
        .delay = 0,
    },
};

static struct v4l2_subdev_info t8es9_subdev_info[] = {
	{
		.code   = V4L2_MBUS_FMT_SBGGR10_1X10,
		.colorspace = V4L2_COLORSPACE_JPEG,
		.fmt    = 1,
		.order    = 0,
	},
};

static const struct i2c_device_id t8es9_i2c_id[] = {
	{T8ES9_SENSOR_NAME, (kernel_ulong_t)&t8es9_s_ctrl},
	{ }
};

static int32_t msm_t8es9_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    return msm_sensor_i2c_probe(client, id, &t8es9_s_ctrl);
}

static struct i2c_driver t8es9_i2c_driver = {
	.id_table = t8es9_i2c_id,
	.probe  = msm_t8es9_i2c_probe,
	.driver = {
		.name = T8ES9_SENSOR_NAME,
	},
};

static struct msm_camera_i2c_client t8es9_sensor_i2c_client = {
	.addr_type = MSM_CAMERA_I2C_WORD_ADDR,
};

static const struct of_device_id t8es9_dt_match[] = {
	{.compatible = "qcom,t8es9", .data = &t8es9_s_ctrl},
	{}
};

MODULE_DEVICE_TABLE(of, t8es9_dt_match);

static struct platform_driver t8es9_platform_driver = {
	.driver = {
		.name = "qcom,t8es9",
		.owner = THIS_MODULE,
		.of_match_table = t8es9_dt_match,
	},
};

struct t8es9_data
{
    uint16_t devaddr;
    uint16_t addr;
    uint16_t datalen;
    uint8_t data[4];
};
#define T8ES9_IOC_MAGIC             'T'
#define T8ES9_IOC_I2C_READ _IOWR(T8ES9_IOC_MAGIC, 1, struct t8es9_data)
#define T8ES9_IOC_I2C_WRITE _IOWR(T8ES9_IOC_MAGIC, 2, struct t8es9_data)
static int t8es9_ioc_i2c_read(struct t8es9_data *data)
{
    int32_t rc = 0;
    uint16_t devaddr = data->devaddr ? (data->devaddr >> 1) : (t8es9_sensor_i2c_client.client->addr >> 1);
    uint8_t addr[2];
    struct i2c_msg msgs[2];
    addr[0] = (data->addr & 0xFF00) >> 8;
    addr[1] = (data->addr & 0xFF);
    msgs[0].addr = devaddr;
    msgs[0].flags = 0;
    msgs[0].len = 2;
    msgs[0].buf = addr;
    msgs[1].addr = devaddr;
    msgs[1].flags = I2C_M_RD;
    msgs[1].len = data->datalen <= sizeof(data->data) ? data->datalen : sizeof(data->data);
    msgs[1].buf = (unsigned char *)data->data;
    rc = i2c_transfer(t8es9_sensor_i2c_client.client->adapter, msgs, 2);
    pr_debug("t8es9_ioc_i2c_read(0x%02X, 0x%04X, %d, { 0x%02X, 0x%02X, 0x%02X, 0x%02X } == %d\n",
        devaddr, data->addr, data->datalen, data->data[0], data->data[1], data->data[2], data->data[3], rc);
    if(rc < 0)
        return rc;
    return 0;
}
static int t8es9_ioc_i2c_write(struct t8es9_data *data)
{
    int32_t rc = 0;
    uint16_t devaddr = data->devaddr ? (data->devaddr >> 1) : (t8es9_sensor_i2c_client.client->addr >> 1);
    uint8_t buf[sizeof(data->data) + 2];
    uint16_t len = (data->datalen <= sizeof(data->data) ? data->datalen : sizeof(data->data)) + 2;
    struct i2c_msg msgs[1];
    int i;
    buf[0] = (data->addr & 0xFF00) >> 8;
    buf[1] = (data->addr & 0xFF);
    for(i = 0; i < len - 2; i++)
    {
        buf[2 + i] = data->data[len - 3 - i];
    }
    msgs[0].addr = devaddr;
    msgs[0].flags = 0;
    msgs[0].len = len;
    msgs[0].buf = buf;
    rc = i2c_transfer(t8es9_sensor_i2c_client.client->adapter, msgs, 1);
    pr_debug("t8es9_ioc_i2c_write(0x%02X, 0x%04X, %d, { 0x%02X, 0x%02X, 0x%02X, 0x%02X } == %d\n",
        devaddr, data->addr, data->datalen, buf[0], buf[1], buf[2], buf[3], rc);
    if(rc < 0)
        return rc;
    return 0;
}
static long t8es9_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
    long ret = -EINVAL;
    switch (cmd)
    {
        case T8ES9_IOC_I2C_READ:
        {
            struct t8es9_data data;
            if(copy_from_user(&data, (void __user *)arg, sizeof(struct t8es9_data)))
                return -EFAULT;
            ret = t8es9_ioc_i2c_read(&data);
            if (ret < 0)
                return ret;
            if(copy_to_user((void __user *)arg, &data, sizeof(struct t8es9_data)))
                return -EFAULT;
            break;
        }
        case T8ES9_IOC_I2C_WRITE:
        {
            struct t8es9_data data;
            if(copy_from_user(&data, (void __user *)arg, sizeof(struct t8es9_data)))
                return -EFAULT;
            ret = t8es9_ioc_i2c_write(&data);
            if (ret < 0)
                return ret;
            if(copy_to_user((void __user *)arg, &data, sizeof(struct t8es9_data)))
                return -EFAULT;
            break;
        }
        default:
        {
            printk(KERN_ERR "t8es9_ioctl : unknown cmd %x\n", cmd);
            break;
        }
    }
    return ret;
}

static int32_t t8es9_platform_probe(struct platform_device *pdev)
{
	int32_t rc = 0;
	const struct of_device_id *match;
	match = of_match_device(t8es9_dt_match, &pdev->dev);
	rc = msm_sensor_platform_probe(pdev, match->data);
	return rc;
}

static int __init t8es9_init_module(void)
{
    int32_t rc;
    rc = misc_register(&t8es9_device.dev);
    if(unlikely(rc))
    {
        pr_err("%s:%d failed to register misc device!\n", __func__, __LINE__);
        return rc;
    }

    pr_info("%s:%d\n", __func__, __LINE__);
	rc = platform_driver_probe(&t8es9_platform_driver,
		t8es9_platform_probe);
	if (!rc)
		return rc;
	pr_err("%s:%d rc %d\n", __func__, __LINE__, rc);
	return i2c_add_driver(&t8es9_i2c_driver);
}

static void __exit t8es9_exit_module(void)
{
    int32_t rc;
    rc = misc_deregister(&t8es9_device.dev);
    if (unlikely(rc))
    {
        printk(KERN_ERR "t8es9_device: failed to unregister misc device!\n");
    }
    printk(KERN_INFO "t8es9_device: unloaded\n");
    if (t8es9_s_ctrl.pdev) {
        msm_sensor_free_sensor_data(&t8es9_s_ctrl);
        platform_driver_unregister(&t8es9_platform_driver);
    } else
        i2c_del_driver(&t8es9_i2c_driver);
    return;
}

static struct msm_sensor_ctrl_t t8es9_s_ctrl = {
	.sensor_i2c_client = &t8es9_sensor_i2c_client,
	.power_setting_array.power_setting = t8es9_power_setting,
	.power_setting_array.size = ARRAY_SIZE(t8es9_power_setting),
	.msm_sensor_mutex = &t8es9_mut,
	.sensor_v4l2_subdev_info = t8es9_subdev_info,
	.sensor_v4l2_subdev_info_size = ARRAY_SIZE(t8es9_subdev_info),
};

module_init(t8es9_init_module);
module_exit(t8es9_exit_module);
MODULE_DESCRIPTION("t8es9");
MODULE_LICENSE("GPL v2");
