/* Copyright (c) 2014, Lytro, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/input.h>
#include <linux/gpio_wheels.h>
#include <linux/workqueue.h>
#include <linux/gpio.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>
#include <linux/spinlock.h>

struct gpio_wheel_data {
	const struct gpio_wheel *wheel;
	struct input_dev *input;
	struct timer_list timer;
	struct work_struct work;
	unsigned int timer_debounce;	/* in msecs */
	unsigned int irq_out;
	unsigned int irq_in;
	spinlock_t lock;
	bool disabled;
	bool key_pressed;
	unsigned int isr_num;
	int state_num;
	int wheel_states[2];
};

struct gpio_keys_drvdata {
	struct input_dev *input;
	struct mutex disable_lock;
	unsigned int n_wheels;
	int (*enable)(struct device *dev);
	void (*disable)(struct device *dev);
	struct gpio_wheel_data data[0];
};

static void check_state(int wheel_states[2], unsigned int* state_num){
	*state_num = (wheel_states[0] ? (wheel_states[1] ? 4 : 3 ) : (wheel_states[1] ? 1 : 2));
}

static int probe_state(struct gpio_wheel_data *bdata){
	return	printk("GPIO states : %d-%d(#)\n", bdata->wheel_states[0], bdata->wheel_states[1]);
}


static void reset_state(struct gpio_wheel_data *bdata){
	unsigned long flags;
	spin_lock_irqsave(&bdata->lock, flags);
	bdata->wheel_states[0] = gpio_get_value(bdata->wheel->gpio_out);
	bdata->wheel_states[1] = gpio_get_value(bdata->wheel->gpio_in);
	check_state(bdata->wheel_states, &bdata->state_num);
	spin_unlock_irqrestore(&bdata->lock, flags);
	probe_state(bdata);
}

static ssize_t wheel_states(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct gpio_keys_drvdata *ddata = platform_get_drvdata(pdev);
	int i;
	for(i = 0; i < ddata->n_wheels; i++){
		struct gpio_wheel_data *bdata = &ddata->data[i];
		simple_strtoul(buf,NULL,0) ? probe_state(bdata) : reset_state(bdata);
	}
	return count;
}

static DEVICE_ATTR(recovery_wheel, S_IWUSR , NULL, wheel_states);

static struct attribute *gpio_keys_attrs[] = {
	&dev_attr_recovery_wheel.attr,
	NULL,
};

static struct attribute_group gpio_keys_attr_group = {
	.attrs = gpio_keys_attrs,
};

static void gpio_keys_gpio_report_event(struct gpio_wheel_data *bdata)
{
	const struct gpio_wheel *wheel = bdata->wheel;
	struct input_dev *input = bdata->input;
	unsigned int type = wheel->type ?: EV_KEY;
	int diff, pre_state,  dirt;

	if(bdata->isr_num == 0){
		/* The critical section here is locked at the gpio_keys_probe function. */
		bdata->wheel_states[0] = (gpio_get_value(wheel->gpio_out) ? 1 : 0);
		bdata->wheel_states[1] = (gpio_get_value(wheel->gpio_in) ? 1 : 0);
		check_state(bdata->wheel_states, &bdata->state_num);
	}else{
		/* The critical section is locked at the gpio_keys_isr_out or gpio_keys_isr_out_isr_in functions.*/
		if (type == EV_KEY) {
			bdata->wheel_states[0] = (gpio_get_value(wheel->gpio_out) ? 1 : 0);
			bdata->wheel_states[1] = (gpio_get_value(wheel->gpio_in) ? 1 : 0);
			pre_state = bdata->state_num;
			check_state(bdata->wheel_states, &bdata->state_num);
			diff = bdata->state_num - pre_state;
			if(diff == 1 || diff == -3)
				dirt = 1;
			else if(diff == -1 || diff == 3)
				dirt = -1;
			else
				dirt = 0;
			if(dirt != 0 ){
				input_event(input, type, (dirt > 0 ? wheel->code_clockwise : wheel->code_counter_clockwise), 1);
				input_event(input, type, (dirt > 0 ? wheel->code_clockwise : wheel->code_counter_clockwise), 0);
				input_sync(input);
			}else{
				check_state(bdata->wheel_states, &bdata->state_num);
				printk("Non Mutual exclusion\n");
			}
		}else{
			printk("Not an EV_KEY event.\n");
		}
	}
}

static void gpio_keys_gpio_work_func(struct work_struct *work)
{
	struct gpio_wheel_data *bdata =
		container_of(work, struct gpio_wheel_data, work);

	gpio_keys_gpio_report_event(bdata);
}

static void gpio_keys_gpio_timer(unsigned long _data)
{
	struct gpio_wheel_data *bdata = (struct gpio_wheel_data *)_data;
	schedule_work(&bdata->work);
}


static irqreturn_t gpio_keys_gpio_isr_out(int irq, void *dev_id)
{
	struct gpio_wheel_data *bdata = dev_id;
	unsigned long flags;
	BUG_ON(irq != bdata->irq_out);
	spin_lock_irqsave(&bdata->lock, flags);
	bdata->isr_num = 1;
	if (bdata->timer_debounce){
		mod_timer(&bdata->timer, jiffies + msecs_to_jiffies(bdata->timer_debounce));
	}else{
		schedule_work(&bdata->work);
	}
	spin_unlock_irqrestore(&bdata->lock, flags);
	return IRQ_HANDLED;
}


static irqreturn_t gpio_keys_gpio_isr_in(int irq, void *dev_id)
{
	struct gpio_wheel_data *bdata = dev_id;
	unsigned long flags;
	BUG_ON(irq != bdata->irq_in);
	spin_lock_irqsave(&bdata->lock, flags);
	bdata->isr_num = 2;
	if (bdata->timer_debounce){
		mod_timer(&bdata->timer, jiffies + msecs_to_jiffies(bdata->timer_debounce));
	}else{
		schedule_work(&bdata->work);
	}
	spin_unlock_irqrestore(&bdata->lock, flags);
	return IRQ_HANDLED;
}

static int __devinit gpio_keys_setup_key(struct platform_device *pdev,
					 struct input_dev *input,
					 struct gpio_wheel_data *bdata,
					 const struct gpio_wheel *wheel)
{
	const char *desc = wheel->desc ? wheel->desc : "gpio_keys";
	struct device *dev = &pdev->dev;
	irq_handler_t isr_out, isr_in;
	unsigned long irqflags;
	int irq_out, irq_in, error;
	bdata->input = input;
	bdata->wheel = wheel;
	spin_lock_init(&bdata->lock);
	if (gpio_is_valid(wheel->gpio_out) && gpio_is_valid(wheel->gpio_in)) {

		error = gpio_request(wheel->gpio_out, desc);
		if (error < 0) {
			dev_err(dev, "Failed to request GPIO-OUT %d, error %d\n",
				wheel->gpio_out, error);
			return error;
		}
		error = gpio_request(wheel->gpio_in, desc);
		if (error < 0) {
			dev_err(dev, "Failed to request GPIO-IN %d, error %d\n",
				wheel->gpio_in, error);
			return error;
		}
		error = gpio_direction_input(wheel->gpio_out);
		if (error < 0) {
			dev_err(dev,
				"Failed to configure direction for GPIO-OUT %d, error %d\n",
				wheel->gpio_out, error);
			goto fail;
		}
		error = gpio_direction_input(wheel->gpio_in);
		if (error < 0) {
			dev_err(dev,
				"Failed to configure direction for GPIO-IN %d, error %d\n",
				wheel->gpio_in, error);
			goto fail;
		}
		if (wheel->debounce_interval) {
			error = gpio_set_debounce(wheel->gpio_out,	wheel->debounce_interval * 1000);
			if (error < 0)
				bdata->timer_debounce =	wheel->debounce_interval;

			error = gpio_set_debounce(wheel->gpio_in,	wheel->debounce_interval * 1000);
			if (error < 0)
				bdata->timer_debounce =	wheel->debounce_interval;
		}
		irq_out = gpio_to_irq(wheel->gpio_out);
		if (irq_out < 0) {
			error = irq_out;
			dev_err(dev,
				"Unable to get irq number for GPIO %d, error %d\n",
				wheel->gpio_out, error);
			goto fail;
		}
		bdata->irq_out = irq_out;
		irq_in = gpio_to_irq(wheel->gpio_in);
		if (irq_in < 0) {
			error = irq_in;
			dev_err(dev,
				"Unable to get irq number for GPIO %d, error %d\n",
				wheel->gpio_in, error);
			goto fail;
		}
		bdata->irq_in = irq_in;
		INIT_WORK(&bdata->work, gpio_keys_gpio_work_func);
		setup_timer(&bdata->timer, gpio_keys_gpio_timer, (unsigned long)bdata);
		isr_out = gpio_keys_gpio_isr_out;
		isr_in = gpio_keys_gpio_isr_in;
		irqflags = IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING;
	}else{
		goto fail;
	}
	bdata->isr_num = 0;
	bdata->state_num = 0;
	input_set_capability(input, wheel->type ?: EV_KEY, wheel->code_clockwise);
	input_set_capability(input, wheel->type ?: EV_KEY, wheel->code_counter_clockwise);

	if (!wheel->can_disable)
		irqflags |= IRQF_SHARED;
	error = request_any_context_irq(bdata->irq_out, isr_out, irqflags, desc, bdata);
	if (error < 0) {
		dev_err(dev, "Unable to claim irq %d; error %d\n",
			bdata->irq_out, error);
		goto fail;
	}

	error = request_any_context_irq(bdata->irq_in, isr_in, irqflags, desc, bdata);
	if (error < 0) {
		dev_err(dev, "Unable to claim irq %d; error %d\n",
			bdata->irq_in, error);
		goto fail;
	}
	return 0;

fail:
	if (gpio_is_valid(wheel->gpio_out))
		gpio_free(wheel->gpio_out);
	if (gpio_is_valid(wheel->gpio_in))
		gpio_free(wheel->gpio_in);
	return error;
}

static int gpio_keys_open(struct input_dev *input)
{
	struct gpio_keys_drvdata *ddata = input_get_drvdata(input);

	return ddata->enable ? ddata->enable(input->dev.parent) : 0;
}

static void gpio_keys_close(struct input_dev *input)
{
	struct gpio_keys_drvdata *ddata = input_get_drvdata(input);

	if (ddata->disable)
		ddata->disable(input->dev.parent);
}

#ifdef CONFIG_OF
static int gpio_keys_get_devtree_pdata(struct device *dev,
			    struct gpio_wheels_platform_data *pdata)
{
	struct device_node *node, *pp;
	int i;
	struct gpio_wheel *wheels;
	u32 reg;
	node = dev->of_node;
	if (node == NULL)
		return -ENODEV;

	memset(pdata, 0, sizeof *pdata);
	pdata->rep = !!of_get_property(node, "autorepeat", NULL);
	pdata->name = of_get_property(node, "input-name", NULL);
	pdata->nwheels = 0;
	pp = NULL;
	while ((pp = of_get_next_child(node, pp)))
		pdata->nwheels++;

	if (pdata->nwheels == 0)
		return -ENODEV;

	wheels = kzalloc(pdata->nwheels * (sizeof *wheels), GFP_KERNEL);
	if (!wheels)
		return -ENOMEM;

	pp = NULL;
	i = 0;
	while ((pp = of_get_next_child(node, pp))) {
		enum of_gpio_flags flags;

		if (!of_find_property(pp, "gpios,outside", NULL)) {
			pdata->nwheels--;
			dev_warn(dev, "Found wheel without gpios-out\n");
			continue;
		}
		wheels[i].gpio_out = of_get_named_gpio_flags(pp, "gpios,outside", 0, &flags);
		if (!of_find_property(pp, "gpios,inside", NULL)) {
			pdata->nwheels--;
			dev_warn(dev, "Found wheel without gpios-in\n");
			continue;
		}
		wheels[i].gpio_in = of_get_named_gpio_flags(pp, "gpios,inside", 0, &flags);
		wheels[i].active_low = flags & OF_GPIO_ACTIVE_LOW;
		if (of_property_read_u32(pp, "linux,code,outside", &reg)) {
			dev_err(dev, "Wheel without keycode: 0x%x\n", wheels[i].code_clockwise);
			goto out_fail;
		}
		wheels[i].code_clockwise = reg;
		if (of_property_read_u32(pp, "linux,code,inside", &reg)) {
			dev_err(dev, "wheel without keycode: 0x%x\n", wheels[i].code_counter_clockwise);
			goto out_fail;
		}
		wheels[i].code_counter_clockwise = reg;
		wheels[i].desc = of_get_property(pp, "label", NULL);
		if (of_property_read_u32(pp, "linux,input-type", &reg) == 0)
			wheels[i].type = reg;
		else
			wheels[i].type = EV_KEY;

		wheels[i].wakeup = !!of_get_property(pp, "gpio-key,wakeup", NULL);

		if (of_property_read_u32(pp, "debounce-interval", &reg) == 0)
			wheels[i].debounce_interval = reg;
		else
			wheels[i].debounce_interval = 5;
		i++;
	}
	pdata->wheels = wheels;
	return 0;
out_fail:
	kfree(wheels);
	return -ENODEV;
}

static struct of_device_id gpio_keys_of_match[] = {
	{ .compatible = "wheel-keys", },
	{ },
};

MODULE_DEVICE_TABLE(of, gpio_keys_of_match);

#else

static int gpio_keys_get_devtree_pdata(struct device *dev,
			    struct gpio_wheels_platform_data *altp)
{
	return -ENODEV;
}

#define gpio_keys_of_match NULL

#endif

static void gpio_remove_key(struct gpio_wheel_data *bdata)
{
	free_irq(bdata->irq_out, bdata);
	free_irq(bdata->irq_in, bdata);
	if (bdata->timer_debounce)
		del_timer_sync(&bdata->timer);
	cancel_work_sync(&bdata->work);
	if (gpio_is_valid(bdata->wheel->gpio_out))
		gpio_free(bdata->wheel->gpio_out);
	if (gpio_is_valid(bdata->wheel->gpio_in))
		gpio_free(bdata->wheel->gpio_in);
}

static int __devinit gpio_keys_probe(struct platform_device *pdev)
{
	const struct gpio_wheels_platform_data *pdata = pdev->dev.platform_data;
	struct gpio_keys_drvdata *ddata;
	struct device *dev = &pdev->dev;
	struct gpio_wheels_platform_data alt_pdata;
	struct input_dev *input;
	int i, error;
	int wakeup = 0;
	unsigned long flags;

	if (!pdata) {
		error = gpio_keys_get_devtree_pdata(dev, &alt_pdata);
		if (error)
			return error;
		pdata = &alt_pdata;
	}

	ddata = kzalloc(sizeof(struct gpio_keys_drvdata) +
			pdata->nwheels * sizeof(struct gpio_wheel_data),
			GFP_KERNEL);
	input = input_allocate_device();
	if (!ddata || !input) {
		dev_err(dev, "failed to allocate state\n");
		error = -ENOMEM;
		goto fail1;
	}

	ddata->input = input;
	ddata->n_wheels = pdata->nwheels;
	ddata->enable = pdata->enable;
	ddata->disable = pdata->disable;
	mutex_init(&ddata->disable_lock);
	platform_set_drvdata(pdev, ddata);
	input_set_drvdata(input, ddata);
	input->name = pdata->name ? : pdev->name;
	input->phys = "wheel-keys/input0";
	input->dev.parent = &pdev->dev;
	input->open = gpio_keys_open;
	input->close = gpio_keys_close;
	input->id.bustype = BUS_HOST;
	input->id.vendor = 0x0001;
	input->id.product = 0x0001;
	input->id.version = 0x0100;
	if (pdata->rep)
		__set_bit(EV_REP, input->evbit);
	for (i = 0; i < pdata->nwheels; i++) {
		const struct gpio_wheel *wheel = &pdata->wheels[i];
		struct gpio_wheel_data *bdata = &ddata->data[i];
		error = gpio_keys_setup_key(pdev, input, bdata, wheel);
		if (error)
			goto fail2;

		if (wheel->wakeup)
			wakeup = 1;
	}
	error = sysfs_create_group(&pdev->dev.kobj, &gpio_keys_attr_group);
	if (error) {
		dev_err(dev, "Unable to export keys/switches, error: %d\n",
			error);
		goto fail2;
	}
	error = input_register_device(input);
	if (error) {
		dev_err(dev, "Unable to register input device, error: %d\n",
			error);
		goto fail3;
	}
	for (i = 0; i < pdata->nwheels; i++) {
		struct gpio_wheel_data *bdata = &ddata->data[i];
		spin_lock_irqsave(&bdata->lock, flags);
		if (gpio_is_valid(bdata->wheel->gpio_out) && gpio_is_valid(bdata->wheel->gpio_in))
			gpio_keys_gpio_report_event(bdata);
		spin_unlock_irqrestore(&bdata->lock, flags);
	}
	input_sync(input);
	device_init_wakeup(&pdev->dev, wakeup);
	return 0;
 fail3:
	sysfs_remove_group(&pdev->dev.kobj, &gpio_keys_attr_group);
 fail2:
	while (--i >= 0)
		gpio_remove_key(&ddata->data[i]);

	platform_set_drvdata(pdev, NULL);
 fail1:
	input_free_device(input);
	kfree(ddata);
	if (!pdev->dev.platform_data)
		kfree(pdata->wheels);

	return error;
}

static int __devexit gpio_keys_remove(struct platform_device *pdev)
{
	struct gpio_keys_drvdata *ddata = platform_get_drvdata(pdev);
	struct input_dev *input = ddata->input;
	int i;

	sysfs_remove_group(&pdev->dev.kobj, &gpio_keys_attr_group);

	device_init_wakeup(&pdev->dev, 0);

	for (i = 0; i < ddata->n_wheels; i++)
		gpio_remove_key(&ddata->data[i]);

	input_unregister_device(input);

	if (!pdev->dev.platform_data)
		kfree(ddata->data[0].wheel);

	kfree(ddata);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int gpio_keys_suspend(struct device *dev)
{
	struct gpio_keys_drvdata *ddata = dev_get_drvdata(dev);
	int i;

	if (device_may_wakeup(dev)) {
		for (i = 0; i < ddata->n_wheels; i++) {
			struct gpio_wheel_data *bdata = &ddata->data[i];
			if (bdata->wheel->wakeup){
				enable_irq_wake(bdata->irq_out);
				enable_irq_wake(bdata->irq_in);
			}
		}
	}

	return 0;
}

static int gpio_keys_resume(struct device *dev)
{
	struct gpio_keys_drvdata *ddata = dev_get_drvdata(dev);
	int i;

	for (i = 0; i < ddata->n_wheels; i++) {
		struct gpio_wheel_data *bdata = &ddata->data[i];
		if (bdata->wheel->wakeup && device_may_wakeup(dev)){
			disable_irq_wake(bdata->irq_out);
			disable_irq_wake(bdata->irq_in);
		}

		if (gpio_is_valid(bdata->wheel->gpio_out) && gpio_is_valid(bdata->wheel->gpio_in))
			gpio_keys_gpio_report_event(bdata);
	}
	input_sync(ddata->input);

	return 0;
}
#endif

static SIMPLE_DEV_PM_OPS(gpio_keys_pm_ops, gpio_keys_suspend, gpio_keys_resume);

static struct platform_driver gpio_keys_device_driver = {
	.probe		= gpio_keys_probe,
	.remove		= __devexit_p(gpio_keys_remove),
	.driver		= {
		.name	= "gpio-keys-wheel",
		.owner	= THIS_MODULE,
		.pm	= &gpio_keys_pm_ops,
		.of_match_table = gpio_keys_of_match,
	}
};

static int __init gpio_keys_init(void)
{
	return platform_driver_register(&gpio_keys_device_driver);
}

static void __exit gpio_keys_exit(void)
{
	platform_driver_unregister(&gpio_keys_device_driver);
}

late_initcall(gpio_keys_init);
module_exit(gpio_keys_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Wheel driver for GPIOs");
MODULE_ALIAS("platform:gpio-keys-wheel");
